#include <gst/gst.h>
#include <gst/video/videooverlay.h>
#import "GStreamerBackend.h"
#import "GStreamerDisplayBackend.h"
#import "VideoStreamBackendDelegate.h"
#import "VSLogging.h"
#import "EaglUIVIew.h"

@implementation GStreamerDisplayBackend
{
    NSUInteger *_windowHandleSetOccurences;
    BOOL _needsRestart;
    CGRect _nextDisplayFrame;
}

@synthesize
displayView = _displayView,
firstTouchPosition,
displayCenterAtFirstTouch;

#pragma mark public interface methods

-(id) initWithDelegate:(id)delegate streamId:(NSString *)streamId displayView:(EaglUIView *)displayView
{
    if (self = [super initWithDelegate:delegate streamId:streamId])
    {
        // we need to set EAGLContext synchronously for each display
        // if we go through the paused state first, we should be ok to set EAGLContext and window handle
        // and paused state does not need data coming in (compared to play state) so we dont need to wait
        // for an unknown amount of time if there is a problem with the stream
        _waitForPipelineStart = YES;
        _transitionPipelineThroughPausedState = YES;
        
        _displayView = displayView;
        _windowHandleSetOccurences = 0;
    }
    return self;
}

-(void) start
{
    TRC_ENTRY;
    if (_pipelineIsRunning)
    {
        // if pipeline is running, just set it to the playing state
        if (gst_element_set_state(_pipeline, GST_STATE_PLAYING) == GST_STATE_CHANGE_FAILURE)
        {
            LOG_ERR(@"Unable to set running display pipeline in PLAYING state");
        }
        else
        {
            [self setActive:YES];
        }
    }
    else
    {
        // if pipeline isn't running, initialize and run it
        LOG_DEBUG(@"Issuing full pipeline start for display backend");
        [super start];
    }
    TRC_EXIT;
}

-(void) stop
{
    TRC_ENTRY;
    if (_pipelineIsRunning)
    {
        // if the pipeline is running, set it to the paused state
        if (gst_element_set_state(_pipeline, GST_STATE_PAUSED) == GST_STATE_CHANGE_FAILURE)
        {
            LOG_ERR(@"Unable top set running display pipeline in READY state");
        }
        else
        {
            [self setActive:NO];
        }
    }
    TRC_EXIT;
}

-(void) restartWithFrame:(CGRect)frame
{
    _needsRestart = YES;
    _nextDisplayFrame = frame;
    [super stop];
}

-(void) pipelineDidStop
{
    LOG_DEBUG(@"*** pipeline did stop!");
    if (_needsRestart)
    {
        _needsRestart = NO;
        dispatch_async(dispatch_get_main_queue(), ^{
            _displayView.frame = _nextDisplayFrame;
            [super start];
        });
    }
}

#pragma mark private utility methods

/**
 * Override parent method for building pipeline
 */
-(NSString *) buildPipeline
{
    TRC_ENTRY;
    NSString *initialError = [self buildPipelineUsingParse];
    TRC_EXIT;
    return initialError;
}

-(void) setEaglContext
{
    [_displayView makeContextCurrent];
}

/**
 * When the pipeline is rebuilt, we need a new UIView for the open gl rendering.
 * Handle this internally to rebuild a new view to match the existing one.
 */
-(void) rebuildUIView
{
    LOG_DEBUG(@"*** rebuilding display backend UIView");
    
    dispatch_sync(dispatch_get_main_queue(), ^{
        
        EaglUIView *currentView = _displayView;
        _displayView = [[EaglUIView alloc] initWithFrame:currentView.frame];
        
        // if our view has gesture recognizers, add them to the new view
        if (currentView.gestureRecognizers.count > 0)
        {
            for (UIGestureRecognizer *recognizer in currentView.gestureRecognizers)
            {
                [currentView removeGestureRecognizer:recognizer];
                [_displayView addGestureRecognizer:recognizer];
            }
        }
        
        // if view is showing, show our new view as well
        if (currentView.superview != nil)
        {
            UIView *superView = currentView.superview;
            [currentView removeFromSuperview];
            [superView addSubview:_displayView];
        }
    });
}

/**
 * Build a pipeline using the parse launch method
 */
-(NSString *) buildPipelineUsingParse
{
    TRC_ENTRY;
    NSString *initialError = nil;
    
    // pipeline def
    NSString *source = @"videotestsrc is-live=1 horizontal-speed=1 name=zeVideoSource";
    source = [source stringByAppendingString:@" ! video/x-raw, width=640, height=480"];
    source = [source stringByAppendingString:@" ! videoconvert"];
    source = [source stringByAppendingString:@" ! autovideosink sync=false"];
    
    // build pipeline
    GError *pipelineErr = NULL;
    _pipeline = gst_parse_launch([source cStringUsingEncoding:NSUTF8StringEncoding], &pipelineErr);
    
    // check for error
    if (pipelineErr != NULL)
    {
        initialError = @"Unable to build pipeline using gst_parse_launch";
        LOG_ERR(initialError, nil);
        
        gchar *errMsg = g_strdup_printf ("Pipeline build error: %s", pipelineErr->message);
        LOG_ERR([NSString stringWithUTF8String:errMsg], nil);
        g_free(errMsg);
    }
    
    TRC_EXIT;
    return initialError;
}

/**
 * Callback when our pipeline is "started" (first pipeline state set successfully)
 * which allows us to set our window handle at the right time.
 */
-(void) pipelineStartRequestIssuedSuccessfully
{
    GstElement *videoSink = gst_bin_get_by_interface(GST_BIN(_pipeline), GST_TYPE_VIDEO_OVERLAY);
    if (!videoSink)
    {
        LOG_ERR(@"Could not retrieve overlay video sink");
    }
    else
    {
        // reset the current eaglcontext, to force gsteglglessink to use a specific context
        // this works to run multiple video windows simultaneously, athough care must be taken
        // not to start all video displays at once.  instead they need to be started synchronously.
        // also, in testing, after many stop/starts in which the pipeline is re-created an open gl
        // error is eventually thrown in glDrawElements
        __weak GStreamerDisplayBackend *weakSelf = self;
        dispatch_sync(dispatch_get_main_queue(), ^{
            [weakSelf setEaglContext];
        });
        
        if (_windowHandleSetOccurences > 0)
            [self rebuildUIView];
        
        guintptr videoHandle = (guintptr)(id)_displayView;
        gst_video_overlay_set_window_handle(GST_VIDEO_OVERLAY(videoSink), videoHandle);
        g_object_unref(videoSink);
        
        _windowHandleSetOccurences++;
    }
}

@end
